// Filename: Unio.C
//
// Description:
//   Universal I/O driver for NT VERSION 1.00
//
// A.Kolegov (NIIATT) 30.08.1996

#include "Unio.H"


// global parameters
ULONG PortBase  = 0;           // Port location, in NT's address form
ULONG PortCount = 0;           // Count of contiguous I/O ports


// Driver entry point
//
// Parameters:
//   DriverObject - pointer to driver object created by the system
//   RegistryPath
//
// Returned:
//   STATUS_SUCCESS if the driver initialized correctly, otherwise an error
//   indicating the reason for failure
//

NTSTATUS DriverEntry( IN PDRIVER_OBJECT DriverObject,
                      IN PUNICODE_STRING RegistryPath )
{
  PHYSICAL_ADDRESS PortAddress;
  PLOCAL_DEVICE_INFO pLocalInfo;  // Device extension:
                                  //      local information for each device.
  NTSTATUS Status;
  PDEVICE_OBJECT DeviceObject;
  CM_RESOURCE_LIST ResourceList;  // resource usage list to report to system
  BOOLEAN ResourceConflict;       // true if I/O ports conflict detected
  ULONG DefaultBase;
  ULONG DefaultCount;

  PortBase  = DefaultBase  = UNIO_BASE_PORT;
  PortCount = DefaultCount = UNIO_NUMBER_PORTS;

  // Try to retrieve base I/O port and range from the Parameters key
  // If there isn't anything specified then use the default values
  {
    static WCHAR             SubKeyString[] = L"\\Parameters";
    UNICODE_STRING           paramPath;
    RTL_QUERY_REGISTRY_TABLE paramTable[ 3 ];

    paramPath.MaximumLength = RegistryPath->Length + sizeof( SubKeyString );
    paramPath.Buffer = ExAllocatePool( PagedPool, paramPath.MaximumLength );

    if( paramPath.Buffer != NULL ) {
      RtlMoveMemory( paramPath.Buffer, RegistryPath->Buffer, RegistryPath->Length );
      RtlMoveMemory( &paramPath.Buffer[ RegistryPath->Length / 2 ],
                     SubKeyString, sizeof( SubKeyString ) );
      paramPath.Length = paramPath.MaximumLength - 2;
      RtlZeroMemory( &paramTable[ 0 ], sizeof( paramTable ) );

      paramTable[ 0 ].Flags = RTL_QUERY_REGISTRY_DIRECT;
      paramTable[ 0 ].Name = L"IoPortAddress";
      paramTable[ 0 ].EntryContext = &PortBase;
      paramTable[ 0 ].DefaultType = REG_DWORD;
      paramTable[ 0 ].DefaultData = &DefaultBase;
      paramTable[ 0 ].DefaultLength = sizeof( ULONG );

      paramTable[ 1 ].Flags = RTL_QUERY_REGISTRY_DIRECT;
      paramTable[ 1 ].Name = L"IoPortCount";
      paramTable[ 1 ].EntryContext = &PortCount;
      paramTable[ 1 ].DefaultType = REG_DWORD;
      paramTable[ 1 ].DefaultData = &DefaultCount;
      paramTable[ 1 ].DefaultLength = sizeof(ULONG);

      if( !NT_SUCCESS( RtlQueryRegistryValues(
              RTL_REGISTRY_ABSOLUTE | RTL_REGISTRY_OPTIONAL,
              paramPath.Buffer, &paramTable[ 0 ], NULL, NULL ) ) )
      {
        PortBase  = DefaultBase;
        PortCount = DefaultCount;
      }

      ExFreePool( paramPath.Buffer );
    }
  }

  PortAddress.LowPart  = PortBase;
  PortAddress.HighPart = 0;


  // Register resource usage (ports)
  RtlZeroMemory( (PVOID)&ResourceList, sizeof( ResourceList ) );

  ResourceList.Count = 1;
  ResourceList.List[ 0 ].InterfaceType = Isa;
  // ResourceList.List[ 0 ].Busnumber = 0;             Already 0
  ResourceList.List[ 0 ].PartialResourceList.Count = 1;
  ResourceList.List[ 0 ].PartialResourceList.PartialDescriptors[ 0 ].Type =
                                             CmResourceTypePort;
  ResourceList.List[ 0 ].PartialResourceList.PartialDescriptors[ 0 ].ShareDisposition =
                                             CmResourceShareDriverExclusive;
  ResourceList.List[ 0 ].PartialResourceList.PartialDescriptors[ 0 ].Flags =
                                             CM_RESOURCE_PORT_IO;
  ResourceList.List[ 0 ].PartialResourceList.PartialDescriptors[ 0 ].u.Port.Start =
                                             PortAddress;
  ResourceList.List[ 0 ].PartialResourceList.PartialDescriptors[ 0 ].u.Port.Length =
                                             PortCount - 1;

  // Report our resource usage and detect conflicts
  Status = IoReportResourceUsage(
      NULL,                            // driver class name
      DriverObject,                    // driver object
      &ResourceList,                   // driver resource list
      sizeof( ResourceList ),          // driver resource list size
      NULL,                            // device object
      NULL,                            // device resource list
      0,                               // device resource list size
      FALSE,                           // override conflict
      &ResourceConflict                // conflict detected
  );

  if( ResourceConflict )
      Status = STATUS_DEVICE_CONFIGURATION_ERROR;

  if( !NT_SUCCESS( Status ) ) {
    KdPrint( ("Resource reporting problem %8X", Status) );
    return Status;
  }

  // Initialize the driver object dispatch table
  // NT sends requests to these routines
  DriverObject->MajorFunction[ IRP_MJ_CREATE ]          = UnioDispatch;
  DriverObject->MajorFunction[ IRP_MJ_CLOSE ]           = UnioDispatch;
  DriverObject->MajorFunction[ IRP_MJ_DEVICE_CONTROL ]  = UnioDispatch;
  DriverObject->DriverUnload                            = UnioUnload;

  // Create device
  Status = UnioCreateDevice(
      UNIO_DEVICE_NAME,
      UNIO_TYPE,
      DriverObject,
      &DeviceObject
  );

  if( NT_SUCCESS( Status ) ) {
    PHYSICAL_ADDRESS MappedAddress;
    ULONG MemType;

    // Convert the IO port address into a form NT likes.
    MemType = 1;                       // located in IO space
    HalTranslateBusAddress(
        Isa,                           // interface type
        0,                             // bus number
        PortAddress,                   // bus address
        &MemType,                      // address space
        &MappedAddress                 // translated address
    );


    // Initialize the local driver info for each device object
    pLocalInfo = (PLOCAL_DEVICE_INFO)DeviceObject->DeviceExtension;

    if( MemType == 0 ) {
      // Port is accessed through memory space - so get a virtual address
      pLocalInfo->PortWasMapped = TRUE;

      // BUGBUG
      // MmMapIoSpace can fail if we run out of PTEs, we should be
      // checking the return value here
      pLocalInfo->PortBase = MmMapIoSpace( MappedAddress, PortCount, FALSE );
    }
    else {
      pLocalInfo->PortWasMapped = FALSE;
      pLocalInfo->PortBase = (PVOID)MappedAddress.LowPart;
    }

    pLocalInfo->DeviceObject   = DeviceObject;
    pLocalInfo->DeviceType     = UNIO_TYPE;
    pLocalInfo->PortCount      = PortCount;
    pLocalInfo->PortMemoryType = MemType;
  }
  else {
    // Error creating device - release resources
    RtlZeroMemory( (PVOID)&ResourceList, sizeof( ResourceList ) );

    // Unreport our resource usage
    Status = IoReportResourceUsage(
        NULL,                          // driver class name
        DriverObject,                  // driver object
        &ResourceList,                 // driver resource list
        sizeof( ResourceList ),        // driver resource list size
        NULL,                          // device object
        NULL,                          // device resource list
        0,                             // device resource list size
        FALSE,                         // override conflict
        &ResourceConflict              // conflict detected
    );
  }

  return Status;
}



// Create the device object and the symbolic link in \DosDevices.
// A symbolic link must be created between the device name and an entry
// in \DosDevices in order to allow Win32 applications to open the device.
//
// Parameters:
//   PrototypeName - Name base, # WOULD be appended to this
//   DeviceType    - Type of device to create
//   DriverObject  - Pointer to driver object created by the system
//   ppDevObj      - Pointer to place to store pointer to created device object
//
// Returned:
//   STATUS_SUCCESS if the device and link are created correctly, otherwise
//   an error indicating the reason for failure.
//

NTSTATUS UnioCreateDevice( IN  PWSTR          PrototypeName,
                           IN  DEVICE_TYPE    DeviceType,
                           IN  PDRIVER_OBJECT DriverObject,
                           OUT PDEVICE_OBJECT *ppDevObj )
{
  NTSTATUS Status;                       // Status of utility calls
  UNICODE_STRING NtDeviceName;
  UNICODE_STRING Win32DeviceName;

  // Get UNICODE name for device.
  RtlInitUnicodeString( &NtDeviceName, PrototypeName );

  // Create it
  Status = IoCreateDevice(
      DriverObject,                    // driver object
      sizeof( LOCAL_DEVICE_INFO ),     // device extension size
      &NtDeviceName,                   // device name
      DeviceType,                      // device type
      0,                               // device characteristics
      FALSE,                           // not exclusive
      ppDevObj                         // device object
  );

  if( !NT_SUCCESS( Status ) )
      return Status;                   // give up if create failed

  // Clear local device info memory
  RtlZeroMemory( (*ppDevObj)->DeviceExtension, sizeof( LOCAL_DEVICE_INFO ) );

  // Set up the rest of the device info
  // These are used for IRP_MJ_READ and IRP_MJ_WRITE which we don't use
  //
  // (*ppDevObj)->Flags |= DO_BUFFERED_IO;
  // (*ppDevObj)->AlignmentRequirement = FILE_BYTE_ALIGNMENT;

  RtlInitUnicodeString( &Win32DeviceName, DOS_DEVICE_NAME );

  Status = IoCreateSymbolicLink( &Win32DeviceName, &NtDeviceName );

  // If we couldn't create the link then abort installation
  if( !NT_SUCCESS( Status ) )
      IoDeleteDevice( *ppDevObj );

  return Status;
}


// It is responsible for processing the IRPs
//
// Parameters:
//   pDO  - Pointer to device object
//   pIrp - Pointer to the current IRP
//
// Returned:
//   STATUS_SUCCESS if the IRP was processed successfully, otherwise an error
//   indicating the reason for failure.

NTSTATUS UnioDispatch( IN PDEVICE_OBJECT pDO, IN PIRP pIrp )
{
  PLOCAL_DEVICE_INFO pLDI;
  PIO_STACK_LOCATION pIrpStack;
  NTSTATUS Status;

  // Initialize the irp info field.
  // This is used to return the number of bytes transfered.
  pIrp->IoStatus.Information = 0;

  // Get local info struct
  pLDI = (PLOCAL_DEVICE_INFO)pDO->DeviceExtension;

  pIrpStack = IoGetCurrentIrpStackLocation( pIrp );

  // Set default return status
  Status = STATUS_NOT_IMPLEMENTED;

  // Dispatch based on major fcn code.
  switch( pIrpStack->MajorFunction ) {
    case IRP_MJ_CREATE:
    case IRP_MJ_CLOSE:
      Status = STATUS_SUCCESS;
      break;

    case IRP_MJ_DEVICE_CONTROL:
      // Dispatch on IOCTL
      switch( pIrpStack->Parameters.DeviceIoControl.IoControlCode ) {
        case IOCTL_UNIO_READ_PORT_UCHAR:
        case IOCTL_UNIO_READ_PORT_USHORT:
        case IOCTL_UNIO_READ_PORT_ULONG:
          Status = UnioIoctlReadPort(
              pLDI,
              pIrp,
              pIrpStack,
              pIrpStack->Parameters.DeviceIoControl.IoControlCode
          );
          break;

        case IOCTL_UNIO_WRITE_PORT_UCHAR:
        case IOCTL_UNIO_WRITE_PORT_USHORT:
        case IOCTL_UNIO_WRITE_PORT_ULONG:
          Status = UnioIoctlWritePort(
              pLDI,
              pIrp,
              pIrpStack,
              pIrpStack->Parameters.DeviceIoControl.IoControlCode
          );
          break;

        case IOCTL_UNIO_GET_PARAMETERS:
          Status = UnioIoctlGetParams(
              pLDI,
              pIrp,
              pIrpStack,
              pIrpStack->Parameters.DeviceIoControl.IoControlCode
          );
          break;
      }
      break;
  }

  // We're done with I/O request. Record the status of the I/O action.
  pIrp->IoStatus.Status = Status;

  // Don't boost priority when returning since this took little time.
  IoCompleteRequest( pIrp, IO_NO_INCREMENT );

  return Status;
}



// This routine processes the IOCTLs which read from the ports.
//
// Parameters:
//   pLDI      - local device data
//   pIrp      - IO request packet
//   IrpStack  - The current stack location
//   IoctlCode - The ioctl code from the IRP
//
// Returned:
//   STATUS_SUCCESS           - OK
//   STATUS_INVALID_PARAMETER - The buffer sent to the driver
//                              was too small to contain the
//                              port, or the buffer which
//                              would be sent back to the driver
//                              was not a multiple of the data size
//   STATUS_ACCESS_VIOLATION  - An illegal port number was given

NTSTATUS UnioIoctlReadPort( IN PLOCAL_DEVICE_INFO pLDI,
                            IN PIRP pIrp,
                            IN PIO_STACK_LOCATION IrpStack,
                            IN ULONG IoctlCode )
{
                              // NOTE:  Use METHOD_BUFFERED ioctls.
  PULONG pIOBuffer;           // Pointer to transfer buffer
                              //      (treated as an array of longs).
  ULONG InBufferSize;         // Amount of data avail. from caller.
  ULONG OutBufferSize;        // Max data that caller can accept.
  ULONG nPort;                // Port number to read
  ULONG DataBufferSize;

  // Size of buffer containing data from application
  InBufferSize  = IrpStack->Parameters.DeviceIoControl.InputBufferLength;

  // Size of buffer for data to be sent to application
  OutBufferSize = IrpStack->Parameters.DeviceIoControl.OutputBufferLength;

  // NT copies inbuf here before entry and copies this to outbuf after
  // return, for METHOD_BUFFERED IOCTL's.
  pIOBuffer     = (PULONG)pIrp->AssociatedIrp.SystemBuffer;

  // Check to ensure input buffer is big enough to hold a port number and
  // the output buffer is at least as big as the port data width
  switch( IoctlCode ) {
    default:                            // There isn't really any default but
      /* FALL THRU */                   // this will quiet the compiler
    case IOCTL_UNIO_READ_PORT_UCHAR:
      DataBufferSize = sizeof( UCHAR );
      break;
    case IOCTL_UNIO_READ_PORT_USHORT:
      DataBufferSize = sizeof( USHORT );
      break;
    case IOCTL_UNIO_READ_PORT_ULONG:
      DataBufferSize = sizeof( ULONG );
      break;
  }

  if( InBufferSize != sizeof( ULONG ) || OutBufferSize < DataBufferSize )
      return STATUS_INVALID_PARAMETER;

  // Buffers are big enough
  nPort = *pIOBuffer;                   // Get the I/O port number from the buffer

  if( nPort >= pLDI->PortCount ||
      (nPort + DataBufferSize) > pLDI->PortCount ||
      (((ULONG)pLDI->PortBase + nPort) & (DataBufferSize - 1)) != 0 )
          return STATUS_ACCESS_VIOLATION;     // It was not legal


  if( pLDI->PortMemoryType == 1 ) {     // Address is in I/O space
    switch( IoctlCode ) {
      case IOCTL_UNIO_READ_PORT_UCHAR:
        *(PUCHAR)pIOBuffer =
            READ_PORT_UCHAR( (PUCHAR)((ULONG)pLDI->PortBase + nPort) );
        break;
      case IOCTL_UNIO_READ_PORT_USHORT:
        *(PUSHORT)pIOBuffer =
            READ_PORT_USHORT( (PUSHORT)((ULONG)pLDI->PortBase + nPort) );
        break;
      case IOCTL_UNIO_READ_PORT_ULONG:
        *(PULONG)pIOBuffer =
            READ_PORT_ULONG( (PULONG)((ULONG)pLDI->PortBase + nPort) );
        break;
    }
  }
  else {                                // Address is in Memory space
    switch( IoctlCode ) {
      case IOCTL_UNIO_READ_PORT_UCHAR:
        *(PUCHAR)pIOBuffer =
            READ_REGISTER_UCHAR( (PUCHAR)((ULONG)pLDI->PortBase + nPort) );
        break;
      case IOCTL_UNIO_READ_PORT_USHORT:
        *(PUSHORT)pIOBuffer =
            READ_REGISTER_USHORT( (PUSHORT)((ULONG)pLDI->PortBase + nPort) );
        break;
      case IOCTL_UNIO_READ_PORT_ULONG:
        *(PULONG)pIOBuffer =
            READ_REGISTER_ULONG( (PULONG)((ULONG)pLDI->PortBase + nPort) );
        break;
    }
  }

  // Indicate number of bytes read
  pIrp->IoStatus.Information = DataBufferSize;

  return STATUS_SUCCESS;
}



// This routine processes the IOCTLs which write to the ports.
//
// Parameters:
//   pLDI      - our local device data
//   pIrp      - IO request packet
//   IrpStack  - The current stack location
//   IoctlCode - The ioctl code from the IRP
//
// Returned:
//   STATUS_SUCCESS           - OK
//   STATUS_INVALID_PARAMETER - The buffer sent to the driver
//                              was too small to contain the
//                              port, or the buffer which
//                              would be sent back to the driver
//                              was not a multiple of the data size.
//   STATUS_ACCESS_VIOLATION  - An illegal port number was given.

NTSTATUS UnioIoctlWritePort( IN PLOCAL_DEVICE_INFO pLDI,
                             IN PIRP pIrp,
                             IN PIO_STACK_LOCATION IrpStack,
                             IN ULONG IoctlCode )
{
                              // NOTE:  Use METHOD_BUFFERED ioctls.
  PULONG pIOBuffer;           // Pointer to transfer buffer
                              //      (treated as array of longs).
  ULONG InBufferSize ;        // Amount of data avail. from caller.
  ULONG OutBufferSize ;       // Max data that caller can accept.
  ULONG nPort;                // Port number to read or write.
  ULONG DataBufferSize;

  // Size of buffer containing data from application
  InBufferSize  = IrpStack->Parameters.DeviceIoControl.InputBufferLength;

  // Size of buffer for data to be sent to application
  OutBufferSize = IrpStack->Parameters.DeviceIoControl.OutputBufferLength;

  // NT copies inbuf here before entry and copies this to outbuf after return,
  // for METHOD_BUFFERED IOCTL's.
  pIOBuffer     = (PULONG)pIrp->AssociatedIrp.SystemBuffer;

  // We don't return any data on a write port.
  pIrp->IoStatus.Information = 0;

  // Check to ensure input buffer is big enough to hold a port number as well
  // as the data to write.
  //
  // The relative port # is a ULONG, and the data is the type appropriate to
  // the IOCTL.
  switch( IoctlCode ) {
    default:                    // There isn't really any default but
      /* FALL THRU */           // this will quiet the compiler.
    case IOCTL_UNIO_WRITE_PORT_UCHAR:
      DataBufferSize = sizeof( UCHAR );
      break;
    case IOCTL_UNIO_WRITE_PORT_USHORT:
      DataBufferSize = sizeof( USHORT );
      break;
    case IOCTL_UNIO_WRITE_PORT_ULONG:
      DataBufferSize = sizeof( ULONG );
      break;
  }

  if( InBufferSize < (sizeof( ULONG ) + DataBufferSize) )
      return STATUS_INVALID_PARAMETER;

  nPort = *pIOBuffer++;

  if( nPort >= pLDI->PortCount ||
      (nPort + DataBufferSize) > pLDI->PortCount ||
      (((ULONG)pLDI->PortBase + nPort) & (DataBufferSize - 1)) != 0 )
          return STATUS_ACCESS_VIOLATION;     // Illegal port number

  if( pLDI->PortMemoryType == 1 ) {           // Address is in I/O space
    switch( IoctlCode ) {
      case IOCTL_UNIO_WRITE_PORT_UCHAR:
        WRITE_PORT_UCHAR(
            (PUCHAR)((ULONG)pLDI->PortBase + nPort),
            *(PUCHAR)pIOBuffer );
        break;
      case IOCTL_UNIO_WRITE_PORT_USHORT:
        WRITE_PORT_USHORT(
            (PUSHORT)((ULONG)pLDI->PortBase + nPort),
            *(PUSHORT)pIOBuffer );
        break;
      case IOCTL_UNIO_WRITE_PORT_ULONG:
        WRITE_PORT_ULONG(
            (PULONG)((ULONG)pLDI->PortBase + nPort),
            *(PULONG)pIOBuffer );
        break;
    }
  }
  else {                                      // Address is in Memory space
    switch( IoctlCode ) {
      case IOCTL_UNIO_WRITE_PORT_UCHAR:
        WRITE_REGISTER_UCHAR(
            (PUCHAR)((ULONG)pLDI->PortBase + nPort),
            *(PUCHAR)pIOBuffer );
        break;
      case IOCTL_UNIO_WRITE_PORT_USHORT:
        WRITE_REGISTER_USHORT(
            (PUSHORT)((ULONG)pLDI->PortBase + nPort),
            *(PUSHORT)pIOBuffer );
        break;
      case IOCTL_UNIO_WRITE_PORT_ULONG:
        WRITE_REGISTER_ULONG(
            (PULONG)((ULONG)pLDI->PortBase + nPort),
            *(PULONG)pIOBuffer );
        break;
    }
  }

  return STATUS_SUCCESS;
}


// This routine returned driver parameters PortBase & PortCount
//
// Parameters:
//   pLDI      - local device data
//   pIrp      - IO request packet
//   IrpStack  - The current stack location
//   IoctlCode - The ioctl code from the IRP
//
// Returned:
//   STATUS_SUCCESS           - OK
//   STATUS_INVALID_PARAMETER - The buffer sent to the driver
//                              was too small to contain the
//                              port, or the buffer which
//                              would be sent back to the driver
//                              was not a multiple of the data size
//   STATUS_ACCESS_VIOLATION  - An illegal port number was given

NTSTATUS UnioIoctlGetParams( IN PLOCAL_DEVICE_INFO pLDI,
                             IN PIRP pIrp,
                             IN PIO_STACK_LOCATION IrpStack,
                             IN ULONG IoctlCode )
{
                              // NOTE:  Use METHOD_BUFFERED ioctls.
  PUNIO_PARAMS pIOBuffer;     // Pointer to transfer buffer
                              //      (treated as an array of longs).
  ULONG OutBufferSize;        // Max data that caller can accept.
  ULONG DataBufferSize;

  if( PortBase == 0 )         // no parameters !!!
      return STATUS_ACCESS_VIOLATION;

  // Size of buffer for data to be sent to application
  OutBufferSize = IrpStack->Parameters.DeviceIoControl.OutputBufferLength;

  // IO buffer associate
  pIOBuffer     = (PUNIO_PARAMS)pIrp->AssociatedIrp.SystemBuffer;

  // size of parameters = sizeof( PortBase ) + sizeof( PortCount )
  DataBufferSize = sizeof( UNIO_PARAMS );

  // buffer big enough
  if( OutBufferSize < DataBufferSize )
      return STATUS_INVALID_PARAMETER;

  // put parameters to IO buffer
  pIOBuffer->PortBase  = PortBase;
  pIOBuffer->PortCount = PortCount;

  // Indicate number of bytes transfered
  pIrp->IoStatus.Information = DataBufferSize;

  return STATUS_SUCCESS;
}



// This routine prepares our driver to be unloaded.  It is responsible
// for freeing all resources allocated by DriverEntry as well as any
// allocated while the driver was running.  The symbolic link must be
// deleted as well.
//
// Parameters:
//   DriverObject - Pointer to driver object created by the system
//
// Returned:
//   None

VOID UnioUnload( PDRIVER_OBJECT DriverObject )
{
  PLOCAL_DEVICE_INFO pLDI;
  CM_RESOURCE_LIST NullResourceList;
  BOOLEAN ResourceConflict;
  UNICODE_STRING Win32DeviceName;

  // Find our global data
  pLDI = (PLOCAL_DEVICE_INFO)DriverObject->DeviceObject->DeviceExtension;

  // Unmap the ports
  if( pLDI->PortWasMapped )
      MmUnmapIoSpace( pLDI->PortBase, pLDI->PortCount );

  // Report we're not using any hardware.  If we don't do this
  // then we'll conflict with our servis (!) on the next load
  RtlZeroMemory( (PVOID)&NullResourceList, sizeof( NullResourceList ) );

  IoReportResourceUsage(
      NULL,
      DriverObject,
      &NullResourceList,
      sizeof(ULONG),
      NULL,
      NULL,
      0,
      FALSE,
      &ResourceConflict
  );

  // Assume all handles are closed down
  // Delete the things we allocated - devices, symbolic links
  RtlInitUnicodeString( &Win32DeviceName, DOS_DEVICE_NAME );
  IoDeleteSymbolicLink( &Win32DeviceName );
  IoDeleteDevice( pLDI->DeviceObject );
}
